<?php

declare(strict_types=1);

require_once __DIR__ . '/../lib/Env.php';
require_once __DIR__ . '/../lib/Db.php';

Env::load(__DIR__ . '/../.env');

$id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
if ($id <= 0) {
    http_response_code(400);
    echo 'Invalid draft ID';
    exit;
}

$pdo = Db::getConnection();
$stmt = $pdo->prepare('SELECT * FROM drafts WHERE id = :id');
$stmt->execute([':id' => $id]);
$draft = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$draft) {
    http_response_code(404);
    echo 'Draft not found';
    exit;
}

$imageSrc = '';
if (!empty($draft['image_path']) && file_exists($draft['image_path'])) {
    $imageData = file_get_contents($draft['image_path']);
    if ($imageData !== false) {
        $imageSrc = 'data:image/png;base64,' . base64_encode($imageData);
    }
}

$title = $draft['title'] ?: 'Draft Preview';
$metaDescription = $draft['meta_description'] ?? '';
$contentHtml = sanitizeHtml($draft['content_html'] ?? '');
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= htmlspecialchars($title, ENT_QUOTES, 'UTF-8') ?></title>
    <meta name="description" content="<?= htmlspecialchars($metaDescription, ENT_QUOTES, 'UTF-8') ?>">
    <style>
        body {
            font-family: system-ui, sans-serif;
            max-width: 960px;
            margin: 2rem auto;
            padding: 0 1rem;
            line-height: 1.6;
        }

        header img {
            max-width: 100%;
            height: auto;
            display: block;
            margin-bottom: 1rem;
        }

        article h2 {
            margin-top: 2rem;
        }
    </style>
</head>
<body>
    <header>
        <p>Status: <?= htmlspecialchars($draft['status'] ?? 'unknown', ENT_QUOTES, 'UTF-8') ?></p>
        <h1><?= htmlspecialchars($title, ENT_QUOTES, 'UTF-8') ?></h1>
        <p><?= htmlspecialchars($metaDescription, ENT_QUOTES, 'UTF-8') ?></p>
        <?php if ($imageSrc !== ''): ?>
            <img src="<?= $imageSrc ?>" alt="<?= htmlspecialchars($title, ENT_QUOTES, 'UTF-8') ?>">
        <?php endif; ?>
    </header>
    <article>
        <?= $contentHtml ?>
    </article>
</body>
</html>

<?php

function sanitizeHtml(string $html): string
{
    $allowedTags = ['h1', 'h2', 'h3', 'p', 'ul', 'ol', 'li', 'strong', 'em', 'b', 'i', 'a', 'blockquote', 'br'];
    $dom = new DOMDocument('1.0', 'UTF-8');
    libxml_use_internal_errors(true);
    $dom->loadHTML('<?xml encoding="UTF-8"><div>' . $html . '</div>', LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);
    libxml_clear_errors();

    $container = $dom->getElementsByTagName('div')->item(0);
    if ($container === null) {
        return '';
    }

    sanitizeNode($container, $allowedTags);

    $sanitized = '';
    foreach ($container->childNodes as $child) {
        $sanitized .= $dom->saveHTML($child);
    }

    return $sanitized;
}

function sanitizeNode(DOMNode $node, array $allowedTags): void
{
    if ($node instanceof DOMElement) {
        $tagName = strtolower($node->tagName);

        if (!in_array($tagName, $allowedTags, true)) {
            $parent = $node->parentNode;
            if ($parent !== null) {
                while ($node->firstChild) {
                    $parent->insertBefore($node->firstChild, $node);
                }
                $parent->removeChild($node);
                return;
            }
        }

        cleanAttributes($node);
    }

    foreach (iterator_to_array($node->childNodes) as $child) {
        sanitizeNode($child, $allowedTags);
    }
}

function cleanAttributes(DOMElement $element): void
{
    $allowedAttributes = ['href', 'title'];
    foreach (iterator_to_array($element->attributes) as $attr) {
        $name = strtolower($attr->name);
        $value = trim($attr->value);

        if (strpos($name, 'on') === 0) {
            $element->removeAttribute($attr->name);
            continue;
        }

        if (!in_array($name, $allowedAttributes, true)) {
            $element->removeAttribute($attr->name);
            continue;
        }

        if ($name === 'href' && !isValidLink($value)) {
            $element->removeAttribute($attr->name);
        }
    }
}

function isValidLink(string $url): bool
{
    if ($url === '') {
        return false;
    }

    if (str_starts_with($url, '/') || str_starts_with($url, '#')) {
        return true;
    }

    $parts = parse_url($url);
    if ($parts === false || !isset($parts['scheme'])) {
        return false;
    }

    $scheme = strtolower($parts['scheme']);
    return in_array($scheme, ['http', 'https', 'mailto'], true);
}
